/*
Attempts to go to (select) a location in the active document

action.setup:
	- target (string): specifies what we are looking for; options are 'text' or, if unspecified will simply select the source
	- source (string): how to gather the string to search for; options include 'word', 'line', or if unspecified defaults to the current selection
	- trim (bool): setting to true will strip whitespace from the source
	- discardIndent (bool): if true will strip leading indentation characters (only necessary if trim is true)
	- searchString (string): what to search for; $EDITOR_SELECTION will be replaced with the source text
	- regex (bool): if true, searchString will be interpretted as a regular expression
*/

var utils = loadLibrary('tea-utils');

action.performWithContext = function(context, outError) {
	var range = context.selectedRanges[0];
	// Init variables used in various places
	var text = '', source = '', start, length;
	// Grab our source text
	if (action.setup.source) {
		source = action.setup.source.toLowerCase();
		if (source === 'word') {
			var selection = utils.getWordAndRange(context, range, true);
			text = selection.word;
			range = selection.range;
		} else if (source === 'line') {
			range = context.lineStorage.lineRangeForIndex(range.location);
			text = context.string.substringWithRange(range);
		} else if (range.length > 0) {
			text = context.string.substringWithRange(range);
		}
	}
	
	// Trim the source, if necessary
	if (action.setup.trim) {
		var trimmed;
		if (action.setup.discardIndent) {
			trimmed = utils.trim(context, text, false, 'both', false, false);
		} else {
			trimmed = utils.trim(context, text, false, 'end', false, false);
		}
		// Figure out where the trimmed text starts
		start = text.indexOf(trimmed);
		if (start >= 0) {
			start += range.location;
		}
		// Figure out how long our trimmed text is
		length = trimmed.length;
		// Make sure there's no trailing linebreak if we are trimming
		if (source === 'line' && ['\r\n', '\r', '\n'].indexOf(trimmed.charAt(length-1)) >= 0) {
			length = length - 1;
		}
		range = new Range(start, length);
		text = trimmed;
	}
	
	// Find our target if necessary
	if (action.setup.target && text !== '') {
		// Setup our search string, defaulting to just the text
		var search = '';
		if (action.setup.searchString) {
			search = searchString.replace('$EDITOR_SELECTION', text);
		} else {
			search = text;
		}
		// Find the start and end points of the substring
		// Clear the variables first, just in case they were used earlier
		start = null;
		length = null;
		if (action.setup.regex) {
			var regex = new RegExp('(' + search + ')');
			var match = context.string.match(regex);
			if (match !== null) {
				start = match.index;
				length = match[0].length;
			}
		} else {
			start = context.string.indexOf(search);
			if (start >= 0) {
				length = search.length;
			} else {
				start = null;
			}
		}
		// Construct our new target range
		if (start !== null && length !== null) {
			range = new (start, length);
		}
	}
	
	// Select the range
	context.selectedRanges = [range];
	return true;
};
